<?php
include 'config.php';

// Paginación
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$por_pagina = 15;
$inicio = ($pagina - 1) * $por_pagina;

// Filtros
$filtro_tipo = $_GET['tipo'] ?? '';
$filtro_fecha_desde = $_GET['fecha_desde'] ?? '';
$filtro_fecha_hasta = $_GET['fecha_hasta'] ?? '';
$filtro_usuario = $_GET['usuario'] ?? '';

// Construir consulta con filtros
$where = "1=1";
$params = [];

if ($filtro_tipo) {
    $where .= " AND tipo = ?";
    $params[] = $filtro_tipo;
}

if ($filtro_fecha_desde) {
    $where .= " AND DATE(fecha) >= ?";
    $params[] = $filtro_fecha_desde;
}

if ($filtro_fecha_hasta) {
    $where .= " AND DATE(fecha) <= ?";
    $params[] = $filtro_fecha_hasta;
}

if ($filtro_usuario) {
    $where .= " AND usuario = ?";
    $params[] = $filtro_usuario;
}

// Obtener movimientos
try {
    $sql = "SELECT * FROM caja WHERE $where ORDER BY fecha DESC LIMIT $inicio, $por_pagina";
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $movimientos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Total de movimientos para paginación
    $sql_total = "SELECT COUNT(*) as total FROM caja WHERE $where";
    $stmt_total = $conn->prepare($sql_total);
    $stmt_total->execute($params);
    $total_movimientos = $stmt_total->fetch(PDO::FETCH_ASSOC)['total'];
    $total_paginas = ceil($total_movimientos / $por_pagina);
    
    // Totales
    $sql_totales = "SELECT 
                    SUM(CASE WHEN tipo = 'ingreso' THEN monto ELSE 0 END) as total_ingresos,
                    SUM(CASE WHEN tipo = 'egreso' THEN monto ELSE 0 END) as total_egresos
                    FROM caja WHERE $where";
    $stmt_totales = $conn->prepare($sql_totales);
    $stmt_totales->execute($params);
    $totales = $stmt_totales->fetch(PDO::FETCH_ASSOC);
    
    $total_ingresos = $totales['total_ingresos'] ?? 0;
    $total_egresos = $totales['total_egresos'] ?? 0;
    $saldo = $total_ingresos - $total_egresos;
    
    // Obtener lista de usuarios para filtro
    $sql_usuarios = "SELECT DISTINCT usuario FROM caja ORDER BY usuario";
    $usuarios = $conn->query($sql_usuarios)->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    die("Error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Movimientos de Caja</title>
    <style>
        * { box-sizing: border-box; }
        body { font-family: 'Segoe UI', Arial; margin: 0; padding: 20px; background: #f0f2f5; }
        .container { max-width: 1400px; margin: 0 auto; background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 20px rgba(0,0,0,0.1); }
        h2 { color: #2c3e50; text-align: center; margin-bottom: 20px; }
        .filtros { background: #ecf0f1; padding: 20px; border-radius: 8px; margin-bottom: 20px; }
        .filtro-grid { display: grid; grid-template-columns: 1fr 1fr 1fr 1fr auto; gap: 15px; align-items: end; }
        .resumen { display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-bottom: 20px; }
        .resumen-item { padding: 15px; border-radius: 8px; text-align: center; color: white; font-weight: bold; }
        .ingresos { background: linear-gradient(135deg, #27ae60, #2ecc71); }
        .egresos { background: linear-gradient(135deg, #e74c3c, #c0392b); }
        .saldo { background: linear-gradient(135deg, #3498db, #2980b9); }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #bdc3c7; }
        th { background: #34495e; color: white; position: sticky; top: 0; }
        tr:hover { background: #f8f9fa; }
        .badge { padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: bold; }
        .badge-ingreso { background: #d4f8e8; color: #27ae60; }
        .badge-egreso { background: #fde8e8; color: #e74c3c; }
        .paginacion { margin-top: 20px; text-align: center; }
        .paginacion a { display: inline-block; padding: 8px 12px; margin: 0 2px; background: #3498db; color: white; text-decoration: none; border-radius: 4px; }
        .paginacion a:hover { background: #2980b9; }
        .paginacion .actual { background: #2c3e50; }
        .acciones { display: flex; gap: 10px; margin-top: 20px; }
        .btn { padding: 10px 20px; background: #2c3e50; color: white; text-decoration: none; border-radius: 5px; transition: background 0.3s; }
        .btn:hover { background: #34495e; }
        .btn-primary { background: #3498db; }
        .btn-primary:hover { background: #2980b9; }
        .filtro-buttons { display: flex; gap: 10px; }
        @media (max-width: 768px) { 
            .filtro-grid { grid-template-columns: 1fr; }
            .resumen { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h2>📊 MOVIMIENTOS DE CAJA</h2>
        
        <div class="acciones">
            <a href="cajadiaria.php" class="btn">➕ Nuevo Movimiento</a>
            <a href="reportes.php" class="btn btn-primary">📈 Generar Reporte</a>
        </div>

        <!-- Filtros -->
        <div class="filtros">
            <form method="GET" action="">
                <div class="filtro-grid">
                    <div>
                        <label for="tipo">Tipo:</label>
                        <select id="tipo" name="tipo">
                            <option value="">Todos los tipos</option>
                            <option value="ingreso" <?php echo $filtro_tipo == 'ingreso' ? 'selected' : ''; ?>>Ingresos</option>
                            <option value="egreso" <?php echo $filtro_tipo == 'egreso' ? 'selected' : ''; ?>>Egresos</option>
                        </select>
                    </div>
                    <div>
                        <label for="fecha_desde">Fecha Desde:</label>
                        <input type="date" id="fecha_desde" name="fecha_desde" value="<?php echo htmlspecialchars($filtro_fecha_desde); ?>">
                    </div>
                    <div>
                        <label for="fecha_hasta">Fecha Hasta:</label>
                        <input type="date" id="fecha_hasta" name="fecha_hasta" value="<?php echo htmlspecialchars($filtro_fecha_hasta); ?>">
                    </div>
                    <div>
                        <label for="usuario">Usuario:</label>
                        <select id="usuario" name="usuario">
                            <option value="">Todos los usuarios</option>
                            <?php foreach ($usuarios as $user): ?>
                                <option value="<?php echo htmlspecialchars($user['usuario']); ?>" 
                                    <?php echo $filtro_usuario == $user['usuario'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($user['usuario']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="filtro-buttons">
                        <button type="submit" style="padding: 12px 20px; background: #3498db; color: white; border: none; border-radius: 6px; cursor: pointer;">🔍 Filtrar</button>
                        <a href="ver_movimientos.php" style="padding: 12px 20px; background: #95a5a6; color: white; text-decoration: none; border-radius: 6px; display: inline-block;">🔄 Limpiar</a>
                    </div>
                </div>
            </form>
        </div>

        <!-- Resumen -->
        <div class="resumen">
            <div class="resumen-item ingresos">
                <h3>TOTAL INGRESOS</h3>
                <p style="font-size: 24px; margin: 0;">S/ <?php echo number_format($total_ingresos, 2); ?></p>
            </div>
            <div class="resumen-item egresos">
                <h3>TOTAL EGRESOS</h3>
                <p style="font-size: 24px; margin: 0;">S/ <?php echo number_format($total_egresos, 2); ?></p>
            </div>
            <div class="resumen-item saldo">
                <h3>SALDO ACTUAL</h3>
                <p style="font-size: 24px; margin: 0;">S/ <?php echo number_format($saldo, 2); ?></p>
            </div>
        </div>

        <!-- Tabla de movimientos -->
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Fecha y Hora</th>
                    <th>Documento</th>
                    <th>Responsable</th>
                    <th>Descripción</th>
                    <th>Tipo</th>
                    <th>Monto</th>
                    <th>Usuario</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($movimientos as $mov): ?>
                <tr>
                    <td><?php echo $mov['id_caja']; ?></td>
                    <td><?php echo date('d/m/Y H:i', strtotime($mov['fecha'])); ?></td>
                    <td><?php echo htmlspecialchars($mov['documento']); ?></td>
                    <td><?php echo htmlspecialchars($mov['responsable']); ?></td>
                    <td><?php echo htmlspecialchars($mov['descripcion']); ?></td>
                    <td>
                        <span class="badge badge-<?php echo $mov['tipo']; ?>">
                            <?php echo strtoupper($mov['tipo']); ?>
                        </span>
                    </td>
                    <td style="font-weight: bold; color: <?php echo $mov['tipo'] == 'ingreso' ? '#27ae60' : '#e74c3c'; ?>">
                        S/ <?php echo number_format($mov['monto'], 2); ?>
                    </td>
                    <td><?php echo htmlspecialchars($mov['usuario']); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <?php if (empty($movimientos)): ?>
            <p style="text-align: center; color: #7f8c8d; margin: 40px 0;">No se encontraron movimientos</p>
        <?php endif; ?>

        <!-- Paginación -->
        <?php if ($total_paginas > 1): ?>
        <div class="paginacion">
            <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                <a href="?pagina=<?php echo $i; ?>&tipo=<?php echo $filtro_tipo; ?>&fecha_desde=<?php echo $filtro_fecha_desde; ?>&fecha_hasta=<?php echo $filtro_fecha_hasta; ?>&usuario=<?php echo $filtro_usuario; ?>" 
                   class="<?php echo $i == $pagina ? 'actual' : ''; ?>">
                   <?php echo $i; ?>
                </a>
            <?php endfor; ?>
        </div>
        <?php endif; ?>

        <div style="text-align: center; margin-top: 20px; color: #7f8c8d;">
            Mostrando <?php echo count($movimientos); ?> de <?php echo $total_movimientos; ?> movimientos
        </div>
    </div>
</body>
</html>