<?php
// Conexión a la base de datos
$host = "localhost";
$user = "root";
$pass = "";
$dbname = "inventario_db";

$conn = new mysqli($host, $user, $pass, $dbname);

if ($conn->connect_error) {
    die("Error de conexión: " . $conn->connect_error);
}

// Función para buscar productos
function buscarProductos($conn, $termino)
{
    $query = "SELECT id, codigo, nombre, precio, stock_actual 
              FROM producto 
              WHERE (codigo LIKE ? OR nombre LIKE ?) 
              ORDER BY nombre 
              LIMIT 10";

    $termino = "%" . $termino . "%";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("ss", $termino, $termino);
    $stmt->execute();

    return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
}

// Variables
$productos = [];
$resultado_kardex = null;
$termino_busqueda = '';
$mostrar_reporte = false;

// Verificar si estamos mostrando el reporte
if (isset($_GET['mostrar_reporte']) && $_GET['mostrar_reporte'] == '1') {
    $mostrar_reporte = true;
}

// Procesar búsqueda
if (isset($_GET['buscar']) && !empty($_GET['buscar'])) {
    $termino_busqueda = trim($_GET['buscar']);
    $productos = buscarProductos($conn, $termino_busqueda);
}

// Procesar generación de kardex
if (isset($_GET['id_producto']) && !empty($_GET['id_producto'])) {
    $id_producto_actual = intval($_GET['id_producto']);
    $fecha_inicio = !empty($_GET['fecha_inicio']) ? $_GET['fecha_inicio'] : null;
    $fecha_fin = !empty($_GET['fecha_fin']) ? $_GET['fecha_fin'] : null;

    // Obtener información del producto
    $query_producto = "SELECT * FROM producto WHERE id = ?";
    $stmt = $conn->prepare($query_producto);
    $stmt->bind_param("i", $id_producto_actual);
    $stmt->execute();
    $producto_info = $stmt->get_result()->fetch_assoc();

    if ($producto_info) {
        // Obtener compras con campo calculado
        $query_compras = "SELECT c.fecha, c.num_doc, c.proveedor, 
                                 dc.cantidad, dc.precio, 
                                 (dc.precio * dc.cantidad) as subtotal
                          FROM detalle_compras dc
                          INNER JOIN compras c ON dc.id_compra = c.id
                          WHERE dc.id_producto = ?";

        if ($fecha_inicio && $fecha_fin) {
            $query_compras .= " AND DATE(c.fecha) BETWEEN ? AND ?";
        }
        $query_compras .= " ORDER BY c.fecha ASC";

        $stmt = $conn->prepare($query_compras);
        if ($fecha_inicio && $fecha_fin) {
            $stmt->bind_param("iss", $id_producto_actual, $fecha_inicio, $fecha_fin);
        } else {
            $stmt->bind_param("i", $id_producto_actual);
        }
        $stmt->execute();
        $compras = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

        // Obtener ventas con campo calculado
        $query_ventas = "SELECT v.fecha, v.num_doc, v.cliente, 
                                dv.cantidad, dv.precio,
                                (dv.precio * dv.cantidad) as subtotal
                         FROM detalle_ventas dv
                         INNER JOIN ventas v ON dv.id_venta = v.id
                         WHERE dv.id_producto = ?";

        if ($fecha_inicio && $fecha_fin) {
            $query_ventas .= " AND DATE(v.fecha) BETWEEN ? AND ?";
        }
        $query_ventas .= " ORDER BY v.fecha ASC";

        $stmt = $conn->prepare($query_ventas);
        if ($fecha_inicio && $fecha_fin) {
            $stmt->bind_param("iss", $id_producto_actual, $fecha_inicio, $fecha_fin);
        } else {
            $stmt->bind_param("i", $id_producto_actual);
        }
        $stmt->execute();
        $ventas = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

        // Calcular totales
        $total_compras = array_sum(array_column($compras, 'cantidad'));
        $total_ventas = array_sum(array_column($ventas, 'cantidad'));
        $total_subtotal_compras = array_sum(array_column($compras, 'subtotal'));
        $total_subtotal_ventas = array_sum(array_column($ventas, 'subtotal'));

        $resultado_kardex = [
            'producto' => $producto_info,
            'compras' => $compras,
            'ventas' => $ventas,
            'total_compras' => $total_compras,
            'total_ventas' => $total_ventas,
            'total_subtotal_compras' => $total_subtotal_compras,
            'total_subtotal_ventas' => $total_subtotal_ventas,
            'stock_actual' => $producto_info['stock_actual']
        ];

        $mostrar_reporte = true;
    }
}
?>

<!DOCTYPE html>
<html lang="es">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sistema de Kardex</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
            transition: all 0.3s ease;
        }

        /* Estilos para el BUSCADOR */
        .search-section {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
        }

        .search-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
            max-width: 900px;
            width: 100%;
            overflow: hidden;
        }

        .search-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }

        .search-body {
            padding: 40px;
        }

        .result-card {
            border-left: 5px solid #28a745;
            margin-bottom: 15px;
            transition: all 0.3s;
            border-radius: 8px;
            overflow: hidden;
        }

        .result-card:hover {
            transform: translateX(10px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        /* Estilos para el REPORTE */
        .report-section {
            display: none;
            min-height: 100vh;
            padding: 20px;
            background-color: #f8f9fa;
        }

        .report-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            margin-bottom: 30px;
        }

        .report-header {
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            color: white;
            padding: 30px;
        }

        .report-body {
            padding: 30px;
        }

        .producto-info {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
            border-left: 5px solid #007bff;
        }

        .resumen-box {
            background: #e8f4fd;
            border-radius: 10px;
            padding: 25px;
            margin-bottom: 30px;
        }

        .movimiento-compra {
            background-color: rgba(40, 167, 69, 0.08);
        }

        .movimiento-venta {
            background-color: rgba(255, 193, 7, 0.08);
        }

        .total-row {
            background-color: #e3f2fd !important;
            font-weight: bold;
        }

        .btn-back {
            position: fixed;
            bottom: 20px;
            left: 20px;
            z-index: 1000;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.2);
            padding: 12px 25px;
            font-weight: bold;
        }

        .btn-print {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.2);
            padding: 12px 25px;
            font-weight: bold;
        }

        /* Animaciones */
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }

            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Responsive */
        @media (max-width: 768px) {

            .search-section,
            .report-section {
                padding: 10px;
            }

            .search-body,
            .report-body {
                padding: 20px;
            }

            .btn-back,
            .btn-print {
                position: relative;
                bottom: auto;
                left: auto;
                right: auto;
                margin: 10px;
                width: calc(50% - 20px);
                float: left;
            }
        }
    </style>
</head>

<body>
    <?php if (!$mostrar_reporte): ?>
        <!-- SECCIÓN 1: BUSCADOR (se muestra primero) -->
        <section class="search-section" id="searchSection">
            <div class="search-container fade-in">
                <!-- Encabezado -->
                <div class="search-header">
                    <h1 class="mb-3">🔍 Sistema de Kardex</h1>
                    <p class="lead">Busque un producto para generar el reporte</p>
                </div>

                <!-- Cuerpo -->
                <div class="search-body">
                    <!-- Formulario de búsqueda -->
                    <form method="GET" action="" class="mb-4">
                        <div class="mb-4">
                            <label for="buscar" class="form-label h5">Buscar por código o nombre:</label>
                            <div class="input-group input-group-lg">
                                <span class="input-group-text bg-primary text-white">
                                    <i class="fas fa-search"></i>
                                </span>
                                <input type="text" class="form-control" id="buscar" name="buscar"
                                    placeholder="Ejemplo: AMOXI500 o Amoxicilina"
                                    value="<?php echo htmlspecialchars($termino_busqueda); ?>" required autofocus>
                                <button class="btn btn-primary" type="submit">
                                    <i class="fas fa-search"></i> Buscar
                                </button>
                            </div>
                            <div class="text-center mt-2">
                                <small class="text-muted">Presione Enter o haga clic en Buscar</small>
                            </div>
                        </div>
                    </form>

                    <!-- Resultados de búsqueda -->
                    <?php if (!empty($productos)): ?>
                        <div class="mt-4">
                            <h4 class="mb-3 text-center">
                                <i class="fas fa-list-alt"></i>
                                Productos Encontrados (<?php echo count($productos); ?>)
                            </h4>

                            <?php foreach ($productos as $producto): ?>
                                <div class="card result-card">
                                    <div class="card-body">
                                        <div class="row align-items-center">
                                            <div class="col-md-2">
                                                <strong
                                                    class="text-primary"><?php echo htmlspecialchars($producto['codigo']); ?></strong>
                                            </div>
                                            <div class="col-md-4">
                                                <h6 class="mb-1"><?php echo htmlspecialchars($producto['nombre']); ?></h6>
                                                <small class="text-muted">ID: <?php echo $producto['id']; ?></small>
                                            </div>
                                            <div class="col-md-2">
                                                <span class="badge bg-info p-2">
                                                    $<?php echo number_format($producto['precio'], 2); ?>
                                                </span>
                                            </div>
                                            <div class="col-md-2">
                                                <span
                                                    class="badge <?php echo $producto['stock_actual'] > 0 ? 'bg-success' : 'bg-danger'; ?> p-2">
                                                    <i class="fas fa-box"></i> <?php echo $producto['stock_actual']; ?>
                                                </span>
                                            </div>
                                            <div class="col-md-2 text-end">
                                                <!-- Botón para mostrar el reporte (en la misma página) -->
                                                <form method="GET" action="" style="display: inline;">
                                                    <input type="hidden" name="id_producto" value="<?php echo $producto['id']; ?>">
                                                    <input type="hidden" name="mostrar_reporte" value="1">
                                                    <button type="submit" class="btn btn-primary btn-sm">
                                                        <i class="fas fa-chart-bar"></i> Ver Kardex
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>

                    <?php elseif (isset($_GET['buscar'])): ?>
                        <div class="alert alert-warning text-center fade-in">
                            <i class="fas fa-exclamation-triangle fa-2x mb-3"></i>
                            <h4>No se encontraron productos</h4>
                            <p>No hay coincidencias para: <strong>"<?php echo htmlspecialchars($termino_busqueda); ?>"</strong>
                            </p>
                            <p>Intente con otro término de búsqueda.</p>
                        </div>
                    <?php endif; ?>

                    <!-- Instrucciones -->
                    <div class="alert alert-info mt-4">
                        <h5><i class="fas fa-info-circle"></i> ¿Cómo funciona?</h5>
                        <ul class="mb-0">
                            <li>Busque un producto por código o nombre</li>
                            <li>Seleccione el producto deseado</li>
                            <li>El reporte Kardex se mostrará en esta misma página</li>
                            <li>Use el botón "Volver" para realizar una nueva búsqueda</li>
                        </ul>
                    </div>
                </div>

                <!-- Footer -->
                <div class="search-header text-center py-3">
                    <small>Sistema de Kardex v1.0 | Control de inventario</small>
                </div>
            </div>
        </section>
    <?php endif; ?>

    <?php if ($mostrar_reporte && $resultado_kardex): ?>
        <!-- SECCIÓN 2: REPORTE (se muestra después de seleccionar producto) -->
        <section class="report-section fade-in" id="reportSection" style="display: block;">
            <!-- Botones flotantes -->
            <button class="btn btn-secondary btn-back" onclick="volverABuscar()">
                <i class="fas fa-arrow-left"></i> Volver a Buscar
            </button>

            <button class="btn btn-success btn-print" onclick="window.print()">
                <i class="fas fa-print"></i> Imprimir Reporte
            </button>

            <div class="container">
                <div class="report-container">
                    <!-- Encabezado del reporte -->
                    <div class="report-header">
                        <div class="row align-items-center">
                            <div class="col-md-8">
                                <h1 class="mb-2"><i class="fas fa-chart-line"></i> REPORTE KARDEX</h1>
                                <h3 class="mb-0"><?php echo htmlspecialchars($resultado_kardex['producto']['nombre']); ?>
                                </h3>
                                <p class="mb-0">Generado: <?php echo date('d/m/Y H:i:s'); ?></p>
                            </div>
                            <div class="col-md-4 text-end">
                                <div class="bg-white text-dark p-3 rounded d-inline-block">
                                    <h4 class="mb-0 text-primary">
                                        <?php echo htmlspecialchars($resultado_kardex['producto']['codigo']); ?></h4>
                                    <small class="text-muted">Código del producto</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Cuerpo del reporte -->
                    <div class="report-body">
                        <!-- Información del producto -->
                        <div class="producto-info">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5><i class="fas fa-info-circle"></i> Información del Producto</h5>
                                    <p><strong>Descripción:</strong>
                                        <?php echo htmlspecialchars($resultado_kardex['producto']['descripcion']); ?></p>
                                    <p><strong>Precio Actual:</strong>
                                        $<?php echo number_format($resultado_kardex['producto']['precio'], 2); ?></p>
                                </div>
                                <div class="col-md-6">
                                    <h5><i class="fas fa-filter"></i> Período del Reporte</h5>
                                    <p><strong>Fechas:</strong>
                                        <?php if (isset($_GET['fecha_inicio']) && isset($_GET['fecha_fin'])): ?>
                                            <?php echo date('d/m/Y', strtotime($_GET['fecha_inicio'])); ?>
                                            al <?php echo date('d/m/Y', strtotime($_GET['fecha_fin'])); ?>
                                        <?php else: ?>
                                            Todo el período disponible
                                        <?php endif; ?>
                                    </p>
                                    <!-- Formulario para cambiar fechas -->
                                    <form method="GET" action="" class="row g-2">
                                        <input type="hidden" name="id_producto" value="<?php echo $_GET['id_producto']; ?>">
                                        <input type="hidden" name="mostrar_reporte" value="1">

                                        <div class="col-5">
                                            <input type="date" class="form-control form-control-sm" name="fecha_inicio"
                                                value="<?php echo $_GET['fecha_inicio'] ?? date('Y-m-01'); ?>">
                                        </div>
                                        <div class="col-5">
                                            <input type="date" class="form-control form-control-sm" name="fecha_fin"
                                                value="<?php echo $_GET['fecha_fin'] ?? date('Y-m-d'); ?>">
                                        </div>
                                        <div class="col-2">
                                            <button type="submit" class="btn btn-sm btn-primary w-100">
                                                <i class="fas fa-sync-alt"></i>
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>

                        <!-- Resumen estadístico -->
                        <div class="resumen-box">
                            <h4 class="mb-4 text-center"><i class="fas fa-chart-pie"></i> Resumen del Período</h4>
                            <div class="row text-center">
                                <div class="col-md-3 mb-3">
                                    <div class="p-3 rounded bg-success text-white h-100">
                                        <h5>Compras</h5>
                                        <h2><?php echo $resultado_kardex['total_compras']; ?></h2>
                                        <p class="mb-0">
                                            $<?php echo number_format($resultado_kardex['total_subtotal_compras'], 2); ?>
                                        </p>
                                    </div>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <div class="p-3 rounded bg-danger text-white h-100">
                                        <h5>Ventas</h5>
                                        <h2><?php echo $resultado_kardex['total_ventas']; ?></h2>
                                        <p class="mb-0">
                                            $<?php echo number_format($resultado_kardex['total_subtotal_ventas'], 2); ?></p>
                                    </div>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <div class="p-3 rounded bg-warning text-dark h-100">
                                        <h5>Mov. Neto</h5>
                                        <h2><?php echo $resultado_kardex['total_compras'] - $resultado_kardex['total_ventas']; ?>
                                        </h2>
                                        <p class="mb-0">
                                            <?php echo $resultado_kardex['total_compras'] > $resultado_kardex['total_ventas'] ? 'Entrada neta' : 'Salida neta'; ?>
                                        </p>
                                    </div>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <div class="p-3 rounded bg-primary text-white h-100">
                                        <h5>Stock Final</h5>
                                        <h2><?php echo $resultado_kardex['stock_actual']; ?></h2>
                                        <p class="mb-0">Unidades disponibles</p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Movimientos de Compras -->
                        <?php if (!empty($resultado_kardex['compras'])): ?>
                            <div class="mb-5">
                                <h4 class="mb-3 border-bottom pb-2">
                                    <i class="fas fa-shopping-cart text-success"></i>
                                    Movimientos de Compras (<?php echo count($resultado_kardex['compras']); ?>)
                                </h4>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-hover">
                                        <thead class="table-success">
                                            <tr>
                                                <th>Fecha</th>
                                                <th>Documento</th>
                                                <th>Proveedor</th>
                                                <th class="text-end">Cantidad</th>
                                                <th class="text-end">Precio Unit.</th>
                                                <th class="text-end">Subtotal</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $total_cantidad = 0;
                                            $total_monto = 0;
                                            foreach ($resultado_kardex['compras'] as $compra):
                                                $total_cantidad += $compra['cantidad'];
                                                $total_monto += $compra['subtotal'];
                                                ?>
                                                <tr class="movimiento-compra">
                                                    <td><?php echo date('d/m/Y', strtotime($compra['fecha'])); ?></td>
                                                    <td><?php echo htmlspecialchars($compra['num_doc']); ?></td>
                                                    <td><?php echo htmlspecialchars($compra['proveedor']); ?></td>
                                                    <td class="text-end text-success">+<?php echo $compra['cantidad']; ?></td>
                                                    <td class="text-end">$<?php echo number_format($compra['precio'], 2); ?></td>
                                                    <td class="text-end font-weight-bold">
                                                        $<?php echo number_format($compra['subtotal'], 2); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                            <!-- Total -->
                                            <tr class="total-row">
                                                <td colspan="3" class="text-end"><strong>TOTAL COMPRAS:</strong></td>
                                                <td class="text-end"><strong><?php echo $total_cantidad; ?></strong></td>
                                                <td></td>
                                                <td class="text-end">
                                                    <strong>$<?php echo number_format($total_monto, 2); ?></strong></td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Movimientos de Ventas -->
                        <?php if (!empty($resultado_kardex['ventas'])): ?>
                            <div class="mb-5">
                                <h4 class="mb-3 border-bottom pb-2">
                                    <i class="fas fa-cash-register text-warning"></i>
                                    Movimientos de Ventas (<?php echo count($resultado_kardex['ventas']); ?>)
                                </h4>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-hover">
                                        <thead class="table-warning">
                                            <tr>
                                                <th>Fecha</th>
                                                <th>Documento</th>
                                                <th>Cliente</th>
                                                <th class="text-end">Cantidad</th>
                                                <th class="text-end">Precio Unit.</th>
                                                <th class="text-end">Subtotal</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $total_cantidad = 0;
                                            $total_monto = 0;
                                            foreach ($resultado_kardex['ventas'] as $venta):
                                                $total_cantidad += $venta['cantidad'];
                                                $total_monto += $venta['subtotal'];
                                                ?>
                                                <tr class="movimiento-venta">
                                                    <td><?php echo date('d/m/Y', strtotime($venta['fecha'])); ?></td>
                                                    <td><?php echo htmlspecialchars($venta['num_doc']); ?></td>
                                                    <td><?php echo htmlspecialchars($venta['cliente']); ?></td>
                                                    <td class="text-end text-danger">-<?php echo $venta['cantidad']; ?></td>
                                                    <td class="text-end">$<?php echo number_format($venta['precio'], 2); ?></td>
                                                    <td class="text-end font-weight-bold">
                                                        $<?php echo number_format($venta['subtotal'], 2); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                            <!-- Total -->
                                            <tr class="total-row">
                                                <td colspan="3" class="text-end"><strong>TOTAL VENTAS:</strong></td>
                                                <td class="text-end"><strong><?php echo $total_cantidad; ?></strong></td>
                                                <td></td>
                                                <td class="text-end">
                                                    <strong>$<?php echo number_format($total_monto, 2); ?></strong></td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Si no hay movimientos -->
                        <?php if (empty($resultado_kardex['compras']) && empty($resultado_kardex['ventas'])): ?>
                            <div class="alert alert-info text-center">
                                <i class="fas fa-info-circle fa-2x mb-3"></i>
                                <h4>No se encontraron movimientos</h4>
                                <p>Este producto no tiene movimientos de compra o venta en el período seleccionado.</p>
                                <p>Intente cambiar las fechas del filtro.</p>
                            </div>
                        <?php endif; ?>

                        <!-- Resumen final -->
                        <div class="card border-primary">
                            <div class="card-header bg-primary text-white">
                                <h5 class="mb-0"><i class="fas fa-file-alt"></i> Resumen Final</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p><strong>Stock Inicial Teórico:</strong>
                                            <?php echo $resultado_kardex['stock_actual'] + $resultado_kardex['total_ventas'] - $resultado_kardex['total_compras']; ?>
                                            unidades
                                        </p>
                                        <p><strong>Total Entradas (Compras):</strong>
                                            <?php echo $resultado_kardex['total_compras']; ?> unidades
                                            ($<?php echo number_format($resultado_kardex['total_subtotal_compras'], 2); ?>)
                                        </p>
                                        <p><strong>Total Salidas (Ventas):</strong>
                                            <?php echo $resultado_kardex['total_ventas']; ?> unidades
                                            ($<?php echo number_format($resultado_kardex['total_subtotal_ventas'], 2); ?>)
                                        </p>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="alert alert-success">
                                            <h5 class="mb-2">Stock Actual Confirmado</h5>
                                            <h2 class="mb-0 text-center">
                                                <?php echo $resultado_kardex['stock_actual']; ?> unidades
                                            </h2>
                                            <p class="text-center mb-0 mt-2">
                                                Valor aproximado:
                                                $<?php echo number_format($resultado_kardex['stock_actual'] * $resultado_kardex['producto']['precio'], 2); ?>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Footer del reporte -->
                    <div class="report-header text-center py-3">
                        <p class="mb-0">
                            <small>
                                Sistema de Kardex | <?php echo date('d/m/Y H:i:s'); ?> |
                                <button class="btn btn-link text-white p-0" onclick="volverABuscar()">
                                    <u>Volver al buscador</u>
                                </button>
                            </small>
                        </p>
                    </div>
                </div>
            </div>
        </section>
    <?php endif; ?>

    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://kit.fontawesome.com/a076d05399.js" crossorigin="anonymous"></script>

    <script>
        // Función para volver al buscador
        function volverABuscar() {
            // Redirigir a la página sin parámetros
            window.location.href = 'kardex.php';
        }

        // Auto-focus en el campo de búsqueda al cargar
        document.addEventListener('DOMContentLoaded', function () {
            var buscarInput = document.getElementById('buscar');
            if (buscarInput) {
                buscarInput.focus();

                // Seleccionar todo el texto si ya hay algo
                if (buscarInput.value) {
                    buscarInput.select();
                }
            }

            // Validar fechas en el formulario
            var fechaForm = document.querySelector('form[action=""]');
            if (fechaForm) {
                fechaForm.addEventListener('submit', function (e) {
                    var fechaInicio = document.querySelector('input[name="fecha_inicio"]');
                    var fechaFin = document.querySelector('input[name="fecha_fin"]');

                    if (fechaInicio && fechaFin && fechaInicio.value && fechaFin.value) {
                        if (new Date(fechaInicio.value) > new Date(fechaFin.value)) {
                            e.preventDefault();
                            alert('Error: La fecha de inicio no puede ser mayor a la fecha fin');
                            return false;
                        }
                    }
                    return true;
                });
            }
        });

        // Efecto de animación al cargar
        window.onload = function () {
            document.body.style.opacity = '0';
            document.body.style.transition = 'opacity 0.5s';

            setTimeout(function () {
                document.body.style.opacity = '1';
            }, 100);
        };

        // Manejar tecla Enter en el buscador
        document.getElementById('buscar')?.addEventListener('keypress', function (e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                this.form.submit();
            }
        });
    </script>
</body>

</html>

<?php
$conn->close();
?>